import { GoogleGenAI, Type } from "@google/genai";
import { BudgetItem, AIProposalResponse } from "../types";

// Initialize Gemini Client
// The API key is obtained exclusively from the environment variable.
const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });

export const generateProposalWithAI = async (
  items: BudgetItem[], 
  subtotal: number,
  discount: number,
  additions: number,
  total: number
): Promise<AIProposalResponse> => {
  const currencyFormatter = new Intl.NumberFormat('pt-BR', { style: 'currency', currency: 'BRL' });

  const itemsList = items.map(item => 
    `- ${item.quantity}x ${item.productName}: ${currencyFormatter.format(item.unitPrice)} (Total: ${currencyFormatter.format(item.quantity * item.unitPrice)})`
  ).join('\n');

  const prompt = `
    Atue como um assistente comercial profissional e cordial.
    Crie uma mensagem de e-mail ou WhatsApp para enviar a um cliente com o seguinte orçamento:

    Itens:
    ${itemsList}

    Resumo Financeiro:
    Subtotal: ${currencyFormatter.format(subtotal)}
    Descontos: -${currencyFormatter.format(discount)}
    Acréscimos/Taxas: +${currencyFormatter.format(additions)}
    
    Valor Total Final: ${currencyFormatter.format(total)}

    A mensagem deve ser educada, destacar o valor dos produtos e se colocar à disposição para dúvidas.
    Retorne a resposta estritamente em JSON com os campos "subject" (assunto) e "body" (corpo da mensagem).
  `;

  try {
    const response = await ai.models.generateContent({
      model: "gemini-2.5-flash",
      contents: prompt,
      config: {
        responseMimeType: "application/json",
        responseSchema: {
            type: Type.OBJECT,
            properties: {
                subject: { type: Type.STRING },
                body: { type: Type.STRING }
            },
            required: ["subject", "body"]
        }
      }
    });

    const text = response.text;
    if (!text) {
        throw new Error("Resposta vazia da IA.");
    }
    
    return JSON.parse(text) as AIProposalResponse;

  } catch (error) {
    console.error("Erro ao gerar proposta:", error);
    throw new Error("Falha ao se comunicar com a IA. Tente novamente.");
  }
};