import React, { useState } from 'react';
import { X, Copy, Check, Wand2 } from 'lucide-react';
import { AIProposalResponse } from '../types';

interface ProposalModalProps {
  isOpen: boolean;
  onClose: () => void;
  isLoading: boolean;
  data: AIProposalResponse | null;
  error: string | null;
}

export const ProposalModal: React.FC<ProposalModalProps> = ({ isOpen, onClose, isLoading, data, error }) => {
  const [copied, setCopied] = useState(false);

  if (!isOpen) return null;

  const handleCopy = () => {
    if (data) {
      const fullText = `Assunto: ${data.subject}\n\n${data.body}`;
      navigator.clipboard.writeText(fullText);
      setCopied(true);
      setTimeout(() => setCopied(false), 2000);
    }
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/50 backdrop-blur-sm animate-in fade-in duration-200">
      <div className="bg-white rounded-2xl shadow-2xl w-full max-w-2xl overflow-hidden flex flex-col max-h-[90vh]">
        
        {/* Header */}
        <div className="p-6 border-b border-slate-100 flex justify-between items-center bg-indigo-600 text-white">
          <div className="flex items-center gap-3">
            <div className="p-2 bg-white/20 rounded-lg">
                <Wand2 className="w-5 h-5 text-white" />
            </div>
            <h2 className="text-xl font-semibold">Proposta Gerada por IA</h2>
          </div>
          <button 
            onClick={onClose}
            className="p-2 hover:bg-white/20 rounded-full transition-colors"
          >
            <X className="w-5 h-5" />
          </button>
        </div>

        {/* Content */}
        <div className="p-6 overflow-y-auto flex-1">
          {isLoading ? (
            <div className="flex flex-col items-center justify-center py-12 space-y-4">
              <div className="relative w-16 h-16">
                <div className="absolute inset-0 border-4 border-indigo-200 rounded-full"></div>
                <div className="absolute inset-0 border-4 border-indigo-600 rounded-full border-t-transparent animate-spin"></div>
              </div>
              <p className="text-slate-500 font-medium">Analisando itens e redigindo proposta...</p>
            </div>
          ) : error ? (
            <div className="p-4 bg-red-50 text-red-700 rounded-xl border border-red-100">
              <p className="font-medium">Ops! Algo deu errado.</p>
              <p className="text-sm mt-1 opacity-90">{error}</p>
            </div>
          ) : data ? (
            <div className="space-y-6">
              <div className="space-y-2">
                <label className="text-xs font-bold uppercase tracking-wider text-slate-400">Assunto Sugerido</label>
                <div className="p-3 bg-slate-50 rounded-lg border border-slate-200 text-slate-800 font-medium select-all">
                  {data.subject}
                </div>
              </div>
              
              <div className="space-y-2">
                <label className="text-xs font-bold uppercase tracking-wider text-slate-400">Mensagem</label>
                <div className="p-4 bg-slate-50 rounded-lg border border-slate-200 text-slate-700 whitespace-pre-wrap leading-relaxed select-all font-mono text-sm">
                  {data.body}
                </div>
              </div>
            </div>
          ) : null}
        </div>

        {/* Footer */}
        {!isLoading && !error && data && (
          <div className="p-4 border-t border-slate-100 bg-slate-50 flex justify-end gap-3">
            <button
              onClick={onClose}
              className="px-4 py-2 text-slate-600 font-medium hover:bg-slate-200 rounded-lg transition-colors"
            >
              Fechar
            </button>
            <button
              onClick={handleCopy}
              className={`flex items-center gap-2 px-6 py-2 rounded-lg font-semibold text-white transition-all shadow-md ${
                copied ? 'bg-green-600 shadow-green-200' : 'bg-indigo-600 hover:bg-indigo-700 shadow-indigo-200'
              }`}
            >
              {copied ? (
                <>
                  <Check className="w-4 h-4" />
                  Copiado!
                </>
              ) : (
                <>
                  <Copy className="w-4 h-4" />
                  Copiar Texto
                </>
              )}
            </button>
          </div>
        )}
      </div>
    </div>
  );
};