import React from 'react';
import { X, Trash2, FolderOpen, Calendar, ArrowRight, Wallet } from 'lucide-react';
import { SavedBudget } from '../types';

interface BudgetListModalProps {
  isOpen: boolean;
  onClose: () => void;
  savedBudgets: SavedBudget[];
  onLoad: (budget: SavedBudget) => void;
  onDelete: (id: string) => void;
}

export const BudgetListModal: React.FC<BudgetListModalProps> = ({ 
  isOpen, 
  onClose, 
  savedBudgets, 
  onLoad, 
  onDelete 
}) => {
  if (!isOpen) return null;

  const currencyFormatter = new Intl.NumberFormat('pt-BR', {
    style: 'currency',
    currency: 'BRL',
  });

  const calculateTotal = (budget: SavedBudget) => {
    const subtotal = budget.items.reduce((sum, item) => sum + (item.quantity * item.unitPrice), 0);
    const discountVal = parseFloat(budget.discountValue) || 0;
    
    let discountAmount = 0;
    if (discountVal > 0) {
        if (budget.discountType === 'fixed') {
            discountAmount = discountVal;
        } else {
            discountAmount = subtotal * (Math.min(discountVal, 100) / 100);
        }
    }
    
    const additionAmount = parseFloat(budget.additionValue) || 0;
    return subtotal - discountAmount + additionAmount;
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-slate-900/60 backdrop-blur-sm animate-in fade-in duration-200">
      <div className="bg-white rounded-2xl shadow-2xl w-full max-w-2xl overflow-hidden flex flex-col max-h-[85vh]">
        
        {/* Header */}
        <div className="p-5 border-b border-slate-100 flex justify-between items-center bg-white">
          <div className="flex items-center gap-3">
            <div className="p-2.5 bg-indigo-50 rounded-xl">
                <FolderOpen className="w-5 h-5 text-indigo-600" />
            </div>
            <div>
                <h2 className="text-xl font-bold text-slate-800">Meus Orçamentos</h2>
                <p className="text-xs text-slate-500 font-medium">Gerencie seus orçamentos salvos</p>
            </div>
          </div>
          <button 
            onClick={onClose}
            className="p-2 text-slate-400 hover:bg-slate-50 hover:text-slate-600 rounded-full transition-colors"
          >
            <X className="w-5 h-5" />
          </button>
        </div>

        {/* Content */}
        <div className="p-6 overflow-y-auto flex-1 bg-slate-50/50">
          {savedBudgets.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-16 space-y-4 text-center">
              <div className="w-20 h-20 bg-slate-100 rounded-full flex items-center justify-center mb-2 shadow-inner">
                <Wallet className="w-10 h-10 text-slate-300" />
              </div>
              <h3 className="text-slate-700 font-semibold text-lg">Nenhum orçamento salvo</h3>
              <p className="text-slate-500 max-w-xs text-sm">Seus orçamentos salvos aparecerão aqui para acesso rápido e edição.</p>
            </div>
          ) : (
            <div className="space-y-4">
              {savedBudgets.map((budget) => (
                <div 
                    key={budget.id} 
                    className="relative bg-white rounded-xl border border-slate-200 shadow-[0_2px_8px_-2px_rgba(0,0,0,0.05)] hover:shadow-[0_8px_16px_-4px_rgba(0,0,0,0.1)] transition-all duration-300 group overflow-hidden"
                >
                  {/* Decorative Left Border */}
                  <div className="absolute left-0 top-0 bottom-0 w-1.5 bg-indigo-500 transition-transform duration-300 group-hover:scale-y-110"></div>

                  <div className="p-5 pl-7">
                      <div className="flex justify-between items-start mb-3">
                        <div>
                            <h3 className="font-bold text-slate-800 text-lg group-hover:text-indigo-700 transition-colors">{budget.name}</h3>
                            <div className="flex items-center gap-2 text-xs text-slate-500 mt-1.5 bg-slate-100 w-fit px-2 py-1 rounded-md">
                                <Calendar className="w-3.5 h-3.5 text-slate-400" />
                                {new Date(budget.date).toLocaleDateString('pt-BR')} • {new Date(budget.date).toLocaleTimeString('pt-BR', {hour: '2-digit', minute:'2-digit'})}
                            </div>
                        </div>
                        <div className="text-right">
                            <span className="block font-bold text-indigo-600 text-lg">{currencyFormatter.format(calculateTotal(budget))}</span>
                            <span className="text-xs font-medium text-slate-400">{budget.items.length} {budget.items.length === 1 ? 'item' : 'itens'}</span>
                        </div>
                      </div>
                      
                      <div className="flex gap-3 mt-5 pt-4 border-t border-slate-50">
                        <button 
                            onClick={() => { onLoad(budget); onClose(); }}
                            className="flex-1 flex items-center justify-center gap-2 bg-slate-800 hover:bg-indigo-600 text-white py-2.5 rounded-lg text-sm font-semibold transition-all shadow-sm active:scale-[0.98]"
                        >
                            Abrir Orçamento
                            <ArrowRight className="w-4 h-4" />
                        </button>
                        <button 
                            onClick={(e) => {
                                e.stopPropagation();
                                onDelete(budget.id);
                            }}
                            className="px-4 py-2.5 text-slate-400 hover:text-red-600 hover:bg-red-50 rounded-lg transition-colors border border-transparent hover:border-red-100"
                            title="Excluir"
                        >
                            <Trash2 className="w-4 h-4" />
                        </button>
                      </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};