import React, { useState, useEffect, useMemo, useRef } from 'react';
import { Plus, Trash2, Calculator, Package, FileText, Download, ChefHat, Percent, DollarSign, Save, FolderOpen, DownloadCloud } from 'lucide-react';
import { BudgetItem, SavedBudget } from './types';
import { BudgetListModal } from './components/BudgetListModal';

// Helper for generating IDs compatible with all environments
const generateId = () => {
  return Date.now().toString(36) + Math.random().toString(36).substring(2);
};

// Interface for html2pdf window object
declare global {
    interface Window {
        html2pdf: any;
        deferredPrompt: any;
    }
}

const App: React.FC = () => {
  // --- STATE ---
  const [items, setItems] = useState<BudgetItem[]>(() => {
    try {
        const saved = localStorage.getItem('budgetItems');
        return saved ? JSON.parse(saved) : [];
    } catch {
        return [];
    }
  });

  const [budgetName, setBudgetName] = useState<string>(() => {
    return localStorage.getItem('currentBudgetName') || '';
  });

  const [currentBudgetId, setCurrentBudgetId] = useState<string | null>(null);
  
  const [newItemName, setNewItemName] = useState('');
  const [newItemQty, setNewItemQty] = useState<string>('1');
  const [newItemPrice, setNewItemPrice] = useState<string>('');

  const [discountValue, setDiscountValue] = useState<string>(() => localStorage.getItem('currentDiscountValue') || '');
  const [discountType, setDiscountType] = useState<'percent' | 'fixed'>(() => (localStorage.getItem('currentDiscountType') as 'percent' | 'fixed') || 'percent');
  const [additionValue, setAdditionValue] = useState<string>(() => localStorage.getItem('currentAdditionValue') || '');
  
  const [savedBudgets, setSavedBudgets] = useState<SavedBudget[]>(() => {
      try {
          const saved = localStorage.getItem('savedBudgets');
          return saved ? JSON.parse(saved) : [];
      } catch {
          return [];
      }
  });
  const [isBudgetListOpen, setIsBudgetListOpen] = useState(false);
  const [installPrompt, setInstallPrompt] = useState<any>(null);
  const [isGeneratingPdf, setIsGeneratingPdf] = useState(false);

  // --- EFFECTS ---

  // Persistence
  useEffect(() => {
    localStorage.setItem('budgetItems', JSON.stringify(items));
    localStorage.setItem('currentBudgetName', budgetName);
    localStorage.setItem('currentDiscountValue', discountValue);
    localStorage.setItem('currentDiscountType', discountType);
    localStorage.setItem('currentAdditionValue', additionValue);
  }, [items, budgetName, discountValue, discountType, additionValue]);

  useEffect(() => {
      localStorage.setItem('savedBudgets', JSON.stringify(savedBudgets));
  }, [savedBudgets]);

  // PWA Install Prompt Listener
  useEffect(() => {
    const handler = (e: any) => {
      e.preventDefault();
      setInstallPrompt(e);
    };
    window.addEventListener('beforeinstallprompt', handler);
    return () => window.removeEventListener('beforeinstallprompt', handler);
  }, []);

  // --- CALCULATIONS ---
  const subtotal = useMemo(() => {
    return items.reduce((sum, item) => sum + (item.quantity * item.unitPrice), 0);
  }, [items]);

  const discountAmount = useMemo(() => {
    const val = parseFloat(discountValue) || 0;
    if (val <= 0) return 0;
    if (discountType === 'fixed') return val;
    return subtotal * (Math.min(val, 100) / 100);
  }, [subtotal, discountValue, discountType]);

  const additionAmount = useMemo(() => {
      return parseFloat(additionValue) || 0;
  }, [additionValue]);

  const totalAmount = subtotal - discountAmount + additionAmount;

  const currencyFormatter = new Intl.NumberFormat('pt-BR', {
    style: 'currency',
    currency: 'BRL',
  });

  // --- HANDLERS ---
  const handleAddItem = (e?: React.FormEvent) => {
    if (e) e.preventDefault();
    if (!newItemName.trim() || !newItemPrice || parseFloat(newItemPrice) < 0) return;

    const newItem: BudgetItem = {
      id: generateId(),
      productName: newItemName,
      quantity: parseFloat(newItemQty) || 1,
      unitPrice: parseFloat(newItemPrice.replace(',', '.'))
    };

    setItems(prev => [...prev, newItem]);
    setNewItemName('');
    setNewItemQty('1');
    setNewItemPrice('');
    
    const nameInput = document.getElementById('productName');
    if (nameInput) nameInput.focus();
  };

  const handleRemoveItem = (id: string) => {
    setItems(prev => prev.filter(item => item.id !== id));
  };

  const handleUpdateItem = (id: string, field: keyof BudgetItem, value: string | number) => {
     setItems(prev => prev.map(item => {
        if (item.id === id) {
            return { ...item, [field]: value };
        }
        return item;
     }));
  };

  const handleSaveBudget = () => {
      if (items.length === 0) {
          alert("Adicione itens antes de salvar.");
          return;
      }
      
      const nameToSave = budgetName.trim() || `Orçamento ${new Date().toLocaleDateString('pt-BR')}`;
      setBudgetName(nameToSave); 

      if (currentBudgetId) {
          setSavedBudgets(prev => prev.map(b => {
              if (b.id === currentBudgetId) {
                  return {
                      ...b,
                      name: nameToSave,
                      date: new Date().toISOString(),
                      items: items,
                      discountValue,
                      discountType,
                      additionValue
                  };
              }
              return b;
          }));
          alert("Orçamento atualizado!");
      } else {
          const newId = generateId();
          const newBudget: SavedBudget = {
              id: newId,
              name: nameToSave,
              date: new Date().toISOString(),
              items: items,
              discountValue,
              discountType,
              additionValue
          };
          setSavedBudgets(prev => [newBudget, ...prev]);
          setCurrentBudgetId(newId);
          alert("Novo orçamento salvo!");
      }
  };

  const handleLoadBudget = (budget: SavedBudget) => {
      if (items.length > 0 && !window.confirm("Substituir orçamento atual?")) return;
      setItems(budget.items);
      setBudgetName(budget.name);
      setDiscountValue(budget.discountValue);
      setDiscountType(budget.discountType);
      setAdditionValue(budget.additionValue);
      setCurrentBudgetId(budget.id);
  };

  const handleDeleteBudget = (id: string) => {
      if (window.confirm("Excluir este orçamento?")) {
          setSavedBudgets(prev => prev.filter(b => b.id !== id));
          if (currentBudgetId === id) setCurrentBudgetId(null);
      }
  };

  const handleClearAll = () => {
      if (window.confirm('Limpar tudo?')) {
        setItems([]);
        setDiscountValue('');
        setAdditionValue('');
        setBudgetName('');
        setCurrentBudgetId(null);
    }
  };

  const handleInstallClick = async () => {
    if (!installPrompt) return;
    installPrompt.prompt();
    const { outcome } = await installPrompt.userChoice;
    if (outcome === 'accepted') {
      setInstallPrompt(null);
    }
  };

  // --- PDF GENERATION LOGIC ---
  const pdfTemplateRef = useRef<HTMLDivElement>(null);

  const handleDownloadPDF = () => {
    if (!window.html2pdf) {
        alert("Erro: Biblioteca de PDF não carregada. Recarregue a página.");
        return;
    }
    setIsGeneratingPdf(true);
  };

  useEffect(() => {
    if (isGeneratingPdf && pdfTemplateRef.current) {
        // Wait for render to stabilize (ensure overlay is visible)
        setTimeout(() => {
            if (!pdfTemplateRef.current) return;
            
            const element = pdfTemplateRef.current;
            const fileName = budgetName.trim() ? `${budgetName.trim()}.pdf` : 'Orcamento_MaisChurrasco.pdf';

            const opt = {
                margin: 0,
                filename: fileName,
                image: { type: 'jpeg', quality: 0.98 },
                html2canvas: { scale: 2, useCORS: true, scrollY: 0 },
                jsPDF: { unit: 'mm', format: 'a4', orientation: 'portrait' }
            };

            window.html2pdf().set(opt).from(element).save().then(() => {
                setIsGeneratingPdf(false);
            }).catch((err: any) => {
                console.error(err);
                setIsGeneratingPdf(false);
                alert("Erro ao gerar PDF.");
            });
        }, 800);
    }
  }, [isGeneratingPdf, budgetName]);

  return (
    <div className="min-h-screen bg-slate-50 text-slate-900 pb-32">
      
      {/* --- VISIBLE PDF OVERLAY --- */}
      {isGeneratingPdf && (
        <div className="fixed inset-0 z-[100] bg-slate-900/90 flex flex-col items-center justify-start overflow-auto py-10">
            <div className="text-white text-xl font-bold mb-4 animate-pulse">Gerando PDF, aguarde...</div>
            
            {/* 
                Visual Wrapper:
                We apply scaling and shadows here so the user sees a nice preview,
                BUT the actual PDF generator captures the inner div.
            */}
            <div style={{ transform: 'scale(0.9)', transformOrigin: 'top center' }}>
                <div 
                    ref={pdfTemplateRef}
                    className="w-[210mm] bg-white text-black p-12"
                    /* 
                       CRITICAL FIX: 
                       Removed 'min-h-[297mm]' and 'shadow-2xl' from the capture target.
                       Allowing height to be auto prevents a second blank page if content overflows slightly.
                       The paper look is now handled by this white div.
                    */
                >
                    <div className="border-b-2 border-slate-800 pb-4 mb-6 flex justify-between items-end">
                        <div>
                            <h1 className="text-3xl font-bold uppercase tracking-tight">Mais Churrasco</h1>
                            <p className="text-slate-500 text-sm mt-1">Soluções para o seu evento</p>
                        </div>
                        <div className="text-right">
                            <p className="text-sm font-medium text-slate-400">Data de Emissão</p>
                            <p className="text-lg font-bold">{new Date().toLocaleDateString('pt-BR')}</p>
                        </div>
                    </div>

                    <div className="mb-8">
                        <h2 className="text-sm text-slate-400 font-bold uppercase tracking-wider mb-1">Cliente / Evento</h2>
                        <div className="text-xl font-semibold border p-3 rounded-lg border-slate-200 bg-slate-50 block w-full">
                            {budgetName || 'Consumidor Final'}
                        </div>
                    </div>

                    <table className="w-full text-sm mb-8">
                        <thead className="bg-slate-100 border-b border-slate-300">
                            <tr>
                                <th className="py-2 px-3 text-left font-bold text-slate-700">Item</th>
                                <th className="py-2 px-3 text-center font-bold text-slate-700">Qtd</th>
                                <th className="py-2 px-3 text-right font-bold text-slate-700">Unitário</th>
                                <th className="py-2 px-3 text-right font-bold text-slate-700">Total</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-200">
                            {items.map((item, idx) => (
                                <tr key={item.id} className={idx % 2 === 0 ? 'bg-white' : 'bg-slate-50'}>
                                    <td className="py-3 px-3">{item.productName}</td>
                                    <td className="py-3 px-3 text-center">{item.quantity}</td>
                                    <td className="py-3 px-3 text-right">{currencyFormatter.format(item.unitPrice)}</td>
                                    <td className="py-3 px-3 text-right font-medium">{currencyFormatter.format(item.quantity * item.unitPrice)}</td>
                                </tr>
                            ))}
                        </tbody>
                    </table>

                    <div className="flex justify-end">
                        <div className="w-1/2 space-y-2">
                            <div className="flex justify-between py-1 border-b border-slate-100">
                                <span className="text-slate-500">Subtotal:</span>
                                <span className="font-semibold">{currencyFormatter.format(subtotal)}</span>
                            </div>
                            {discountAmount > 0 && (
                                <div className="flex justify-between py-1 border-b border-slate-100 text-red-600">
                                    <span>Desconto:</span>
                                    <span>- {currencyFormatter.format(discountAmount)}</span>
                                </div>
                            )}
                            {additionAmount > 0 && (
                                <div className="flex justify-between py-1 border-b border-slate-100 text-green-600">
                                    <span>Taxas/Frete:</span>
                                    <span>+ {currencyFormatter.format(additionAmount)}</span>
                                </div>
                            )}
                            <div className="flex justify-between py-3 border-t-2 border-slate-800 text-xl font-bold mt-2">
                                <span>Total Final:</span>
                                <span>{currencyFormatter.format(totalAmount)}</span>
                            </div>
                        </div>
                    </div>
                    
                    <div className="mt-12 text-center text-xs text-slate-400 border-t border-slate-100 pt-4">
                        <p>Orçamento gerado pelo App Mais Churrasco</p>
                    </div>
                </div>
            </div>
        </div>
      )}

      {/* Header */}
      <header className="bg-white border-b border-slate-200 sticky top-0 z-30 shadow-sm">
        <div className="max-w-5xl mx-auto px-4 sm:px-6 h-16 flex items-center justify-between">
          <div className="flex items-center gap-3">
            <div className="p-2 bg-red-600 rounded-lg">
                <ChefHat className="w-6 h-6 text-white" />
            </div>
            <h1 className="text-xl font-bold text-slate-900">
              Mais Churrasco
            </h1>
          </div>
          <div className="flex items-center gap-2">
            {installPrompt && (
                <button
                    onClick={handleInstallClick}
                    className="flex items-center gap-2 px-3 py-2 bg-indigo-50 text-indigo-700 hover:bg-indigo-100 rounded-lg transition-colors text-sm font-semibold"
                >
                    <DownloadCloud className="w-4 h-4" />
                    <span className="hidden sm:inline">Instalar App</span>
                </button>
            )}
             <button
                onClick={() => setIsBudgetListOpen(true)}
                className="flex items-center gap-2 px-3 py-2 text-slate-600 hover:bg-slate-100 rounded-lg transition-colors text-sm font-medium"
             >
                <FolderOpen className="w-5 h-5" />
                <span className="hidden sm:inline">Meus Orçamentos</span>
             </button>
            <div className="text-sm font-medium text-slate-500 bg-slate-100 px-3 py-1 rounded-full hidden sm:block">
                {items.length} {items.length === 1 ? 'item' : 'itens'}
            </div>
          </div>
        </div>
      </header>

      <main className="max-w-5xl mx-auto px-4 sm:px-6 py-6 space-y-6">

        {/* Client / Budget Name Section */}
        <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-end">
            <div className="flex-1 w-full">
                <label htmlFor="budgetName" className="block text-xs font-semibold uppercase tracking-wider text-slate-500 mb-1 ml-1">
                    Cliente / Nome do Evento
                </label>
                <input 
                    id="budgetName"
                    type="text"
                    placeholder="Ex: Churrasco do João"
                    value={budgetName}
                    onChange={(e) => setBudgetName(e.target.value)}
                    className="w-full h-12 px-4 rounded-xl border border-slate-200 bg-white focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all placeholder:text-slate-300 font-medium text-lg"
                />
            </div>
            <button
                onClick={handleSaveBudget}
                disabled={items.length === 0}
                className="h-12 w-full sm:w-auto px-6 bg-slate-800 hover:bg-slate-900 text-white font-medium rounded-xl flex items-center justify-center gap-2 transition-all active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed"
            >
                <Save className="w-4 h-4" />
                {currentBudgetId ? "Atualizar" : "Salvar"}
            </button>
        </div>

        {/* Add Item Section */}
        <section className="bg-white rounded-2xl shadow-sm border border-slate-200 overflow-hidden">
          <div className="p-4 sm:p-6 bg-slate-50 border-b border-slate-100 flex items-center gap-2">
            <Package className="w-5 h-5 text-indigo-600" />
            <h2 className="font-semibold text-slate-800">Adicionar Novo Item</h2>
          </div>
          <form onSubmit={handleAddItem} className="p-4 sm:p-6 grid grid-cols-12 gap-4 items-end">
            
            <div className="col-span-12 md:col-span-6">
              <label htmlFor="productName" className="block text-xs font-semibold uppercase tracking-wider text-slate-500 mb-1 ml-1">
                Produto / Serviço
              </label>
              <input
                id="productName"
                type="text"
                placeholder="Ex: Picanha Importada"
                value={newItemName}
                onChange={(e) => setNewItemName(e.target.value)}
                className="w-full h-12 px-4 rounded-xl border border-slate-200 bg-slate-50 focus:bg-white focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all placeholder:text-slate-400"
              />
            </div>

            <div className="col-span-4 md:col-span-2">
              <label htmlFor="qty" className="block text-xs font-semibold uppercase tracking-wider text-slate-500 mb-1 ml-1">
                Qtd
              </label>
              <input
                id="qty"
                type="number"
                min="1"
                step="0.1"
                value={newItemQty}
                onChange={(e) => setNewItemQty(e.target.value)}
                className="w-full h-12 px-4 rounded-xl border border-slate-200 bg-slate-50 focus:bg-white focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all text-center"
              />
            </div>

            <div className="col-span-8 md:col-span-3">
              <label htmlFor="price" className="block text-xs font-semibold uppercase tracking-wider text-slate-500 mb-1 ml-1">
                Preço Unit. (R$)
              </label>
              <input
                id="price"
                type="number"
                placeholder="0,00"
                step="0.01"
                min="0"
                value={newItemPrice}
                onChange={(e) => setNewItemPrice(e.target.value)}
                className="w-full h-12 px-4 rounded-xl border border-slate-200 bg-slate-50 focus:bg-white focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all text-right"
              />
            </div>

            <div className="col-span-12 md:col-span-1 mt-2 md:mt-0">
              <button
                type="submit"
                disabled={!newItemName || !newItemPrice}
                className="w-full h-12 flex items-center justify-center bg-indigo-600 hover:bg-indigo-700 disabled:opacity-50 disabled:cursor-not-allowed text-white rounded-xl shadow-lg shadow-indigo-200 transition-all active:scale-95"
              >
                <Plus className="w-6 h-6" />
              </button>
            </div>
          </form>
        </section>

        {/* Items List */}
        <section className="space-y-4">
          <div className="flex items-center justify-between px-2">
            <h2 className="text-lg font-bold text-slate-800 flex items-center gap-2">
                <FileText className="w-5 h-5 text-slate-400" />
                Itens do Orçamento
            </h2>
            {items.length > 0 && (
                <button 
                    onClick={handleClearAll}
                    className="text-xs font-medium text-red-500 hover:text-red-700 hover:bg-red-50 px-3 py-1.5 rounded-lg transition-colors"
                >
                    Limpar Tudo
                </button>
            )}
          </div>

          {items.length === 0 ? (
            <div className="bg-white border border-dashed border-slate-300 rounded-2xl p-12 text-center">
              <div className="w-16 h-16 bg-slate-50 rounded-full flex items-center justify-center mx-auto mb-4">
                <Calculator className="w-8 h-8 text-slate-300" />
              </div>
              <p className="text-slate-500 font-medium">Nenhum item adicionado ainda.</p>
              <p className="text-slate-400 text-sm mt-1">Preencha os campos acima para começar.</p>
            </div>
          ) : (
            <div className="bg-white rounded-2xl border border-slate-200 shadow-sm overflow-hidden">
                {/* Desktop Header */}
                <div className="hidden md:grid grid-cols-12 gap-4 p-4 bg-slate-50 border-b border-slate-100 text-xs font-semibold uppercase tracking-wider text-slate-500">
                    <div className="col-span-5 pl-2">Produto</div>
                    <div className="col-span-2 text-center">Qtd</div>
                    <div className="col-span-2 text-right">Unitário</div>
                    <div className="col-span-2 text-right">Subtotal</div>
                    <div className="col-span-1 text-center">Ação</div>
                </div>

                <div className="divide-y divide-slate-100">
                  {items.map((item) => (
                    <div key={item.id} className="group md:grid md:grid-cols-12 md:gap-4 p-4 items-center hover:bg-slate-50 transition-colors relative">
                        {/* Mobile View Structure */}
                        <div className="flex justify-between md:hidden mb-2">
                            <span className="font-semibold text-slate-800 text-lg">{item.productName}</span>
                            <button onClick={() => handleRemoveItem(item.id)} className="text-red-400 p-2 -mr-2">
                                <Trash2 className="w-5 h-5" />
                            </button>
                        </div>
                        
                        {/* Product Name */}
                        <div className="md:col-span-5 hidden md:block">
                            <input 
                                type="text"
                                value={item.productName}
                                onChange={(e) => handleUpdateItem(item.id, 'productName', e.target.value)}
                                className="w-full bg-transparent border-none p-2 font-medium text-slate-800 focus:ring-0 placeholder:text-slate-300"
                            />
                        </div>

                        {/* Quantity */}
                        <div className="flex items-center justify-between md:block md:col-span-2">
                             <span className="md:hidden text-sm text-slate-500">Quantidade:</span>
                             <input 
                                type="number"
                                value={item.quantity}
                                min="0"
                                onChange={(e) => handleUpdateItem(item.id, 'quantity', parseFloat(e.target.value) || 0)}
                                className="w-20 md:w-full bg-slate-100 md:bg-transparent border border-slate-200 md:border-none rounded-lg md:rounded-none px-2 py-1 md:p-2 text-right md:text-center text-slate-600 focus:ring-0"
                            />
                        </div>

                        {/* Unit Price */}
                        <div className="flex items-center justify-between md:block md:col-span-2">
                             <span className="md:hidden text-sm text-slate-500">Preço Unitário:</span>
                             <div className="flex items-center justify-end">
                                <span className="text-slate-400 text-sm mr-1 md:hidden">R$</span>
                                <input 
                                    type="number"
                                    value={item.unitPrice}
                                    min="0"
                                    step="0.01"
                                    onChange={(e) => handleUpdateItem(item.id, 'unitPrice', parseFloat(e.target.value) || 0)}
                                    className="w-28 md:w-full bg-slate-100 md:bg-transparent border border-slate-200 md:border-none rounded-lg md:rounded-none px-2 py-1 md:p-2 text-right text-slate-600 focus:ring-0"
                                />
                             </div>
                        </div>

                        {/* Subtotal - Read Only */}
                        <div className="flex items-center justify-between md:block md:col-span-2 mt-2 md:mt-0 pt-2 md:pt-0 border-t border-slate-100 md:border-none">
                            <span className="md:hidden font-semibold text-slate-700">Subtotal:</span>
                            <div className="text-right font-bold text-slate-800">
                                {currencyFormatter.format(item.quantity * item.unitPrice)}
                            </div>
                        </div>

                         {/* Desktop: Delete Action */}
                         <div className="hidden md:flex md:col-span-1 justify-center">
                            <button 
                                onClick={() => handleRemoveItem(item.id)}
                                className="p-2 text-slate-300 hover:text-red-500 hover:bg-red-50 rounded-lg transition-all opacity-0 group-hover:opacity-100"
                                title="Remover item"
                            >
                                <Trash2 className="w-5 h-5" />
                            </button>
                        </div>
                    </div>
                  ))}
                </div>
            </div>
          )}

          {/* Summary / Totals Section */}
          {items.length > 0 && (
            <div className="mt-8 bg-white rounded-2xl border border-slate-200 p-6 space-y-4">
                <h3 className="font-semibold text-slate-800 border-b border-slate-100 pb-2">Resumo do Pedido</h3>
                
                {/* Subtotal */}
                <div className="flex justify-between text-slate-500 text-sm sm:text-base">
                    <span>Subtotal</span>
                    <span>{currencyFormatter.format(subtotal)}</span>
                </div>

                {/* Discount Input */}
                <div className="flex justify-between items-center">
                    <div className="flex items-center gap-2">
                        <span className="text-slate-700 text-sm sm:text-base">Desconto</span>
                        <div className="flex items-center gap-1">
                            <input
                                type="number"
                                min="0"
                                value={discountValue}
                                onChange={(e) => setDiscountValue(e.target.value)}
                                className="w-20 px-2 py-1 text-sm border border-slate-200 rounded-lg focus:outline-none focus:border-indigo-500"
                                placeholder="0"
                            />
                            <div className="flex bg-slate-100 rounded-lg p-0.5 border border-slate-200">
                                <button
                                    onClick={() => setDiscountType('percent')}
                                    className={`p-1 rounded-md transition-all ${discountType === 'percent' ? 'bg-white shadow-sm text-indigo-600' : 'text-slate-400'}`}
                                >
                                    <Percent className="w-3.5 h-3.5" />
                                </button>
                                <button
                                    onClick={() => setDiscountType('fixed')}
                                    className={`p-1 rounded-md transition-all ${discountType === 'fixed' ? 'bg-white shadow-sm text-indigo-600' : 'text-slate-400'}`}
                                >
                                    <DollarSign className="w-3.5 h-3.5" />
                                </button>
                            </div>
                        </div>
                    </div>
                    {discountAmount > 0 && <span className="text-red-600 font-medium">- {currencyFormatter.format(discountAmount)}</span>}
                </div>

                {/* Addition Input */}
                <div className="flex justify-between items-center">
                    <div className="flex items-center gap-2">
                        <span className="text-slate-700 text-sm sm:text-base">Acréscimos (Frete/Taxas)</span>
                        <div>
                             <input
                                type="number"
                                min="0"
                                value={additionValue}
                                onChange={(e) => setAdditionValue(e.target.value)}
                                className="w-20 px-2 py-1 text-sm border border-slate-200 rounded-lg focus:outline-none focus:border-indigo-500"
                                placeholder="0,00"
                            />
                        </div>
                    </div>
                    {additionAmount > 0 && <span className="text-green-600 font-medium">+ {currencyFormatter.format(additionAmount)}</span>}
                </div>

                <div className="border-t border-slate-200 pt-4 flex justify-between items-center text-xl font-bold text-slate-900">
                    <span>Total Final</span>
                    <span>{currencyFormatter.format(totalAmount)}</span>
                </div>
            </div>
          )}
        </section>
      </main>

      {/* Sticky Bottom Summary Bar */}
      <div className="fixed bottom-0 left-0 right-0 bg-white border-t border-slate-200 shadow-[0_-4px_20px_-10px_rgba(0,0,0,0.1)] p-4 sm:p-6 z-20">
        <div className="max-w-5xl mx-auto flex flex-col sm:flex-row items-center justify-between gap-4">
            
            <div className="flex items-center gap-4 w-full sm:w-auto justify-between sm:justify-start">
                <div className="text-sm font-medium text-slate-500 uppercase tracking-wider">Total Final</div>
                <div className="text-2xl font-bold text-indigo-700">
                    {currencyFormatter.format(totalAmount)}
                </div>
            </div>

            <div className="flex gap-3 w-full sm:w-auto">
                <button
                    onClick={handleDownloadPDF}
                    disabled={items.length === 0 || isGeneratingPdf}
                    className="flex-1 flex items-center justify-center gap-2 px-6 py-3 bg-indigo-600 hover:bg-indigo-700 text-white font-bold rounded-xl shadow-lg shadow-indigo-200 transition-all disabled:opacity-50 disabled:cursor-not-allowed active:scale-95"
                >
                    {isGeneratingPdf ? (
                         <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                    ) : (
                         <Download className="w-5 h-5" />
                    )}
                    <span>{isGeneratingPdf ? 'Gerando...' : 'Baixar PDF'}</span>
                </button>
            </div>
        </div>
      </div>

      <BudgetListModal
        isOpen={isBudgetListOpen}
        onClose={() => setIsBudgetListOpen(false)}
        savedBudgets={savedBudgets}
        onLoad={handleLoadBudget}
        onDelete={handleDeleteBudget}
      />
    </div>
  );
};

export default App;