<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0"> <title>Delivery Carnes & Assados</title>
    
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;700&display=swap" rel="stylesheet">

    <style>
        /* --- ESTILOS GERAIS (CSS) --- */
        :root {
            --primary: #FF6B00; /* Laranja Ifood */
            --dark: #2D3436;
            --light: #F9F9F9;
            --grey: #B2BEC3;
        }

        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--light);
            margin: 0; padding: 0; padding-bottom: 80px; /* Espaço pro carrinho */
        }

        /* Header */
        header {
            background: white;
            padding: 15px 20px;
            position: sticky; top: 0; z-index: 100;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            display: flex; justify-content: space-between; align-items: center;
        }
        header h1 { margin: 0; font-size: 1.2rem; color: var(--dark); }
        .btn-local { color: var(--primary); font-size: 0.9rem; border: none; background: none; cursor: pointer; }

        /* Navegação de Categorias */
        .categorias-scroll {
            display: flex; gap: 10px; padding: 15px 20px;
            overflow-x: auto; white-space: nowrap;
            background: var(--light);
        }
        .cat-chip {
            background: white; padding: 8px 15px; border-radius: 20px;
            font-size: 0.9rem; color: var(--dark); border: 1px solid #eee;
            cursor: pointer; transition: 0.2s;
        }
        .cat-chip.ativo { background: var(--primary); color: white; border-color: var(--primary); }

        /* Lista de Produtos */
        .container { padding: 0 20px; }
        
        .section-title { margin-top: 20px; margin-bottom: 10px; font-size: 1.1rem; color: var(--dark); }

        .produto-card {
            background: white; border-radius: 12px; padding: 15px;
            margin-bottom: 15px; display: flex; gap: 15px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.03);
            align-items: center;
        }

        .img-placeholder {
            width: 80px; height: 80px; background: #eee; border-radius: 8px;
            display: flex; align-items: center; justify-content: center; color: #ccc;
            flex-shrink: 0;
        }

        .prod-info { flex: 1; }
        .prod-nome { font-weight: 600; margin-bottom: 5px; color: var(--dark); }
        .prod-desc { font-size: 0.75rem; color: var(--grey); margin-bottom: 8px; }
        .prod-preco { color: var(--primary); font-weight: 700; }

        /* Controles de Quantidade */
        .controles { display: flex; align-items: center; gap: 10px; }
        
        /* Botões +/- */
        .btn-qtd {
            width: 30px; height: 30px; border-radius: 50%; border: 1px solid #ddd;
            background: white; color: var(--primary); font-weight: bold; cursor: pointer;
        }
        
        /* Input para Peso (KG) */
        .input-kg-wrapper { display: flex; align-items: center; background: #f0f0f0; border-radius: 5px; padding: 2px 8px; }
        .input-kg {
            width: 60px; border: none; background: transparent; text-align: center;
            font-weight: bold; font-family: inherit; outline: none;
        }

        /* Barra Inferior (Carrinho) */
        .carrinho-bar {
            position: fixed; bottom: 0; left: 0; width: 100%;
            background: white; padding: 15px 20px;
            box-shadow: 0 -2px 10px rgba(0,0,0,0.1);
            display: flex; justify-content: space-between; align-items: center;
            box-sizing: border-box;
        }
        .total-info { font-size: 0.9rem; color: var(--grey); }
        .total-valor { font-size: 1.2rem; font-weight: 800; color: var(--dark); }
        
        .btn-finalizar {
            background: var(--primary); color: white; border: none;
            padding: 12px 25px; border-radius: 8px; font-weight: 600;
            cursor: pointer;
        }

        /* Modal Simples (Checkout) */
        .modal {
            display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.5); z-index: 200; justify-content: center; align-items: flex-end;
        }
        .modal-content {
            background: white; width: 100%; border-radius: 20px 20px 0 0;
            padding: 25px; box-sizing: border-box; animation: slideUp 0.3s;
        }
        @keyframes slideUp { from {transform: translateY(100%)} to {transform: translateY(0)} }
        
        input.form-control { width: 100%; padding: 10px; margin-bottom: 10px; border: 1px solid #ddd; border-radius: 5px; box-sizing: border-box;}
        
    </style>
</head>
<body>

    <header>
        <div>
            <h1>🍗 Churrasco Delivery</h1>
            <small id="status-loc">📍 Definir local de entrega</small>
        </div>
        <button class="btn-local" onclick="pegarLocalizacao()">
            <i class="fas fa-crosshairs"></i> GPS
        </button>
    </header>

    <div class="categorias-scroll" id="menu-categorias">
        <span class="cat-chip ativo">Todos</span>
    </div>

    <div class="container" id="lista-produtos">
        <p style="text-align:center; padding: 20px;">Carregando cardápio...</p>
    </div>

    <div class="carrinho-bar">
        <div>
            <div class="total-info">Total do Pedido</div>
            <div class="total-valor" id="valor-total">R$ 0,00</div>
        </div>
        <button class="btn-finalizar" onclick="abrirCheckout()">
            Ver Sacola <i class="fas fa-shopping-bag"></i>
        </button>
    </div>

    <div class="modal" id="modal-checkout">
        <div class="modal-content">
            <h2>Finalizar Pedido</h2>
            <p>Confirme seus dados para entrega:</p>
            
            <input type="text" id="cli-nome" class="form-control" placeholder="Seu Nome">
            <input type="tel" id="cli-tel" class="form-control" placeholder="Seu WhatsApp">
            <input type="text" id="cli-end" class="form-control" placeholder="Endereço (Rua, Número)">
            
            <input type="hidden" id="cli-lat">
            <input type="hidden" id="cli-lng">

            <div style="display:flex; gap:10px; margin-top:10px;">
                <button class="btn-finalizar" style="background:#ccc; flex:1;" onclick="fecharCheckout()">Voltar</button>
                <button class="btn-finalizar" style="flex:2;" onclick="enviarPedido()">✅ Enviar Pedido</button>
            </div>
        </div>
    </div>

    <script>
    let cardapio = [];
    let carrinho = {}; // Objeto: { id_produto: {qtd: 1, preco: 10, ...} }

    // 1. Carregar Cardápio ao iniciar
    document.addEventListener("DOMContentLoaded", () => {
        fetch('api/cardapio.php')
            .then(res => res.json())
            .then(data => {
                cardapio = data;
                renderizarCardapio(data);
                renderizarCategorias(data);
            })
            .catch(err => {
                console.error(err);
                alert("Erro ao carregar cardápio. Verifique se api/cardapio.php está funcionando.");
            });
    });

    // 2. Renderizar Categorias (Chips no topo)
    function renderizarCategorias(data) {
        const container = document.getElementById('menu-categorias');
        // Botão 'Todos'
        container.innerHTML = `<span class="cat-chip ativo" onclick="filtrarCat('all', this)">Todos</span>`;
        
        data.forEach(cat => {
            container.innerHTML += `<span class="cat-chip" onclick="filtrarCat(${cat.id}, this)">${cat.titulo}</span>`;
        });
    }

    // --- NOVA FUNÇÃO CORRIGIDA: FILTRAR/ROLAR CATEGORIAS ---
    function filtrarCat(id, elementoHtml) {
        // 1. Visual: Troca a cor do botão ativo
        document.querySelectorAll('.cat-chip').forEach(c => c.classList.remove('ativo'));
        elementoHtml.classList.add('ativo');

        // 2. Comportamento: Rola até a categoria
        if (id === 'all') {
            window.scrollTo({ top: 0, behavior: 'smooth' });
        } else {
            const secao = document.getElementById('cat-' + id);
            if (secao) {
                // Rola a página até o título da categoria, descontando o tamanho do header (80px)
                const y = secao.getBoundingClientRect().top + window.scrollY - 80;
                window.scrollTo({ top: y, behavior: 'smooth' });
            }
        }
    }

    // 3. Renderizar Produtos
    function renderizarCardapio(data) {
        const container = document.getElementById('lista-produtos');
        container.innerHTML = '';

        data.forEach(cat => {
            // Título da Categoria (Importante para o scroll funcionar)
            container.innerHTML += `<div class="section-title" id="cat-${cat.id}">${cat.titulo}</div>`;

            cat.produtos.forEach(prod => {
                let inputHTML = '';
                
                if(prod.unidade === 'KG') {
                    // LÓGICA DE PESO
                    inputHTML = `
                        <div class="input-kg-wrapper">
                            <input type="number" class="input-kg" 
                                step="0.100" min="0" value="0" 
                                placeholder="0.0"
                                onchange="atualizarCarrinho(${prod.id}, '${prod.nome}', ${prod.preco}, this.value, 'KG')">
                            <span style="font-size:0.8rem">kg</span>
                        </div>`;
                } else {
                    // LÓGICA DE UNIDADE
                    inputHTML = `
                        <div class="controles">
                            <button class="btn-qtd" onclick="mudarQtdUnidade(${prod.id}, '${prod.nome}', ${prod.preco}, -1)">-</button>
                            <span id="qtd-display-${prod.id}" style="font-weight:bold; width:20px; text-align:center;">0</span>
                            <button class="btn-qtd" onclick="mudarQtdUnidade(${prod.id}, '${prod.nome}', ${prod.preco}, 1)">+</button>
                        </div>`;
                }

                const card = `
                    <div class="produto-card">
                        <div class="img-placeholder"><i class="fas fa-utensils"></i></div>
                        <div class="prod-info">
                            <div class="prod-nome">${prod.nome}</div>
                            <div class="prod-desc">${prod.descricao || ''}</div>
                            <div class="prod-preco">
                                R$ ${parseFloat(prod.preco).toFixed(2)} 
                                <small style="font-weight:normal; color:#999">/${prod.unidade}</small>
                            </div>
                        </div>
                        <div style="display:flex; flex-direction:column; align-items:end; gap:5px;">
                            ${inputHTML}
                        </div>
                    </div>
                `;
                container.innerHTML += card;
            });
        });
    }

    // --- LÓGICA DO CARRINHO ---

    function mudarQtdUnidade(id, nome, preco, delta) {
        const display = document.getElementById(`qtd-display-${id}`);
        let atual = parseInt(display.innerText);
        let nova = atual + delta;
        
        if (nova < 0) nova = 0;
        display.innerText = nova;

        atualizarCarrinho(id, nome, preco, nova, 'UN');
    }

    function atualizarCarrinho(id, nome, preco, qtd, unidade) {
        qtd = parseFloat(qtd);
        
        // Se a quantidade for zero ou inválida, remove do carrinho
        if (!qtd || qtd <= 0) {
            delete carrinho[id];
        } else {
            carrinho[id] = {
                id_produto: id,
                nome: nome, 
                preco_unitario: preco,
                quantidade: qtd,
                unidade: unidade,
                subtotal: qtd * preco
            };
        }
        recalcularTotal();
    }

    function recalcularTotal() {
        let total = 0;
        Object.values(carrinho).forEach(item => {
            total += item.subtotal;
        });
        
        // Atualiza o texto na barra inferior
        document.getElementById('valor-total').innerText = 'R$ ' + total.toFixed(2).replace('.', ',');
        
        // Animação visual no botão
        const btn = document.querySelector('.btn-finalizar');
        btn.style.transform = "scale(1.05)";
        setTimeout(() => btn.style.transform = "scale(1)", 200);
    }

    // --- GEOLOCALIZAÇÃO ---
    function pegarLocalizacao() {
        const status = document.getElementById('status-loc');
        status.innerText = "⏳ Buscando satélites...";
        
        if (!navigator.geolocation) {
            alert("Seu navegador não suporta geolocalização");
            return;
        }

        navigator.geolocation.getCurrentPosition(
            (pos) => {
                const lat = pos.coords.latitude;
                const lng = pos.coords.longitude;
                
                document.getElementById('cli-lat').value = lat;
                document.getElementById('cli-lng').value = lng;
                
                status.innerText = "✅ Localização definida!";
                status.style.color = "green";
            }, 
            (err) => {
                console.error(err);
                status.innerText = "❌ Erro ao pegar local. Digite o endereço.";
                status.style.color = "red";
            },
            { enableHighAccuracy: true }
        );
    }

    // --- CHECKOUT ---
    function abrirCheckout() {
        if (Object.keys(carrinho).length === 0) {
            alert("Seu carrinho está vazio! Adicione itens.");
            return;
        }
        document.getElementById('modal-checkout').style.display = 'flex';
    }

    function fecharCheckout() {
        document.getElementById('modal-checkout').style.display = 'none';
    }

    // --- FUNÇÃO CORRIGIDA: ENVIAR PEDIDO ---
    function enviarPedido() {
        // 1. Coleta os dados dos inputs
        const nome = document.getElementById('cli-nome').value.trim();
        const tel = document.getElementById('cli-tel').value.trim();
        const end = document.getElementById('cli-end').value.trim();
        
        // Se o usuário não clicou no GPS, envia 0
        const lat = document.getElementById('cli-lat').value || 0;
        const lng = document.getElementById('cli-lng').value || 0;

        // 2. Validação básica
        if(!nome || !tel) {
            alert("Por favor, preencha seu NOME e TELEFONE.");
            return;
        }

        // 3. Recalcula o total direto do objeto (Mais seguro que ler o HTML)
        let totalCalculado = 0;
        Object.values(carrinho).forEach(item => totalCalculado += item.subtotal);

        if(totalCalculado <= 0) {
            alert("Erro: O valor total é zero.");
            return;
        }

        // 4. Monta o pacote de dados
        const payload = {
            cliente: { nome: nome, telefone: tel },
            endereco: { descricao: end, lat: lat, lng: lng },
            total_pagar: totalCalculado, // Valor numérico limpo
            itens: Object.values(carrinho)
        };

        console.log("Enviando pedido:", payload); // Para debug no F12

        // 5. Envia para o PHP
        fetch('api/salvar_pedido.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        })
        .then(res => res.json()) // Tenta ler como JSON
        .then(resp => {
            if(resp.sucesso) {
                alert("✅ Pedido #" + resp.pedido_id + " Enviado com Sucesso!");
                location.reload(); 
            } else {
                alert("Ocorreu um erro ao salvar: " + (resp.erro || "Erro desconhecido"));
            }
        })
        .catch(err => {
            console.error("Erro na requisição:", err);
            alert("Erro de conexão. Verifique o console (F12) para detalhes.");
        });
    }
</script>
</body>
</html>