<?php
/**
 * Analisador de Ações do Mercado Brasileiro (B3)
 * Autor: Gemini AI
 * Descrição: Script para cálculo de risco e retorno de portfólio.
 * Nota: Agora configurado para usar a API Brapi.dev com fallback para simulação.
 */

// --- CLASSE DE ANÁLISE FINANCEIRA ---
class FinancialAnalyzer {
    
    // =========================================================================
    // CONFIGURAÇÃO DA API
    // Cadastre-se em https://brapi.dev para obter sua chave gratuita.
    // =========================================================================
    private $apiKey = 'kvMZKk6T4TpZGUpyxy2whg'; 

    /**
     * Busca dados históricos e informações fundamentais
     * Retorna array com ['history' => dados_grafico, 'info' => dados_empresa]
     */
    public function getHistoricalData($ticker, $days = 252) {
        // Se a chave não foi configurada, usa simulação direto
        if ($this->apiKey === 'kvMZKk6T4TpZGUpyxy2whg' || empty($this->apiKey)) {
            return $this->generateSimulatedData($ticker, $days);
        }

        // Tenta buscar na API
        try {
            // Mapeia dias para range da API (aprox)
            $range = '1y'; 
            if ($days <= 30) $range = '1mo';
            elseif ($days <= 90) $range = '3mo';

            $url = "https://brapi.dev/api/quote/{$ticker}?range={$range}&interval=1d&token={$this->apiKey}";
            
            // Suprime erros visuais do file_get_contents e trata via código
            $response = @file_get_contents($url);
            
            if ($response === false) {
                throw new Exception("Falha na conexão com API");
            }

            $json = json_decode($response, true);

            if (!isset($json['results'][0]['historicalDataPrice'])) {
                throw new Exception("Dados não encontrados no JSON");
            }

            $result = $json['results'][0];
            $apiData = $result['historicalDataPrice'];
            $formattedData = [];

            // Extrai informações da empresa
            $info = [
                'longName' => $result['longName'] ?? $ticker,
                'symbol' => $result['symbol'] ?? $ticker,
                'logourl' => $result['logourl'] ?? 'https://via.placeholder.com/50'
            ];

            foreach ($apiData as $day) {
                // Brapi retorna data em timestamp ou string dependendo do endpoint, garantimos aqui
                $dateVal = $day['date'];
                $dateStr = is_numeric($dateVal) ? date('Y-m-d', $dateVal) : date('Y-m-d', strtotime($dateVal));
                
                // Usamos 'close' (fechamento) para análise
                if (isset($day['close'])) {
                    $formattedData[] = [
                        'date' => $dateStr,
                        'price' => (float)$day['close']
                    ];
                }
            }

            return ['history' => $formattedData, 'info' => $info];

        } catch (Exception $e) {
            // Em caso de erro (limite de API, internet, chave errada), usa simulação
            return $this->generateSimulatedData($ticker, $days);
        }
    }

    // --- MÉTODOS DE SIMULAÇÃO (MANTIDOS COMO BACKUP) ---
    private function generateSimulatedData($ticker, $days) {
        $data = [];
        $currentPrice = 100.0;
        
        $volatilityMap = [
            'PETR4' => 0.025, 'VALE3' => 0.022,
            'ITUB4' => 0.015, 'WEGE3' => 0.018,
            'MGLU3' => 0.035
        ];

        // Simula nomes de empresas para o fallback
        $namesMap = [
            'PETR4' => 'Petróleo Brasileiro S.A. - Petrobras',
            'VALE3' => 'Vale S.A.',
            'ITUB4' => 'Itaú Unibanco Holding S.A.',
            'WEGE3' => 'WEG S.A.',
            'MGLU3' => 'Magazine Luiza S.A.'
        ];
        
        $volatility = $volatilityMap[strtoupper($ticker)] ?? 0.02;
        $trend = 0.0005; 

        $date = new DateTime();
        $date->modify("-$days days");

        for ($i = 0; $i < $days; $i++) {
            $change = $currentPrice * ($trend + ($this->boxMuller() * $volatility));
            $currentPrice += $change;
            if ($currentPrice < 0.01) $currentPrice = 0.01;

            $data[] = [
                'date' => $date->format('Y-m-d'),
                'price' => round($currentPrice, 2)
            ];
            $date->modify('+1 day');
            if ($date->format('N') >= 6) $date->modify('+2 days');
        }

        $info = [
            'longName' => $namesMap[strtoupper($ticker)] ?? "Empresa Simulada $ticker",
            'symbol' => strtoupper($ticker),
            'logourl' => "https://ui-avatars.com/api/?name=$ticker&background=random&color=fff"
        ];

        return ['history' => $data, 'info' => $info];
    }

    private function boxMuller() {
        $u = 0; $v = 0;
        while ($u === 0) $u = mt_rand() / mt_getrandmax();
        while ($v === 0) $v = mt_rand() / mt_getrandmax();
        return sqrt(-2.0 * log($u)) * cos(2.0 * M_PI * $v);
    }

    // --- MÉTODOS DE CÁLCULO (MANTIDOS) ---
    public function calculateMetrics($historicalData) {
        if (empty($historicalData)) return null;

        $prices = array_column($historicalData, 'price');
        $returns = [];
        $count = count($prices);

        if ($count < 2) return null;

        for ($i = 1; $i < $count; $i++) {
            $returns[] = log($prices[$i] / $prices[$i - 1]);
        }

        $averageDailyReturn = count($returns) > 0 ? array_sum($returns) / count($returns) : 0;
        $stdDev = $this->stats_standard_deviation($returns);
        
        $annualizedReturn = $averageDailyReturn * 252;
        $annualizedVol = $stdDev * sqrt(252);

        $totalReturn = ($prices[$count - 1] - $prices[0]) / $prices[0];

        $riskFreeRate = 0.10;
        $sharpeRatio = ($annualizedVol > 0) ? ($annualizedReturn - $riskFreeRate) / $annualizedVol : 0;

        $maxDrawdown = 0;
        $peak = $prices[0];
        foreach ($prices as $p) {
            if ($p > $peak) $peak = $p;
            $drawdown = ($peak - $p) / $peak;
            if ($drawdown > $maxDrawdown) $maxDrawdown = $drawdown;
        }

        return [
            'current_price' => end($prices),
            'total_return' => $totalReturn * 100,
            'volatility' => $annualizedVol * 100, 
            'sharpe' => $sharpeRatio,             
            'max_drawdown' => $maxDrawdown * 100, 
            'data_points' => $historicalData
        ];
    }

    private function stats_standard_deviation(array $a, $sample = false) {
        $n = count($a);
        if ($n === 0) return 0;
        $mean = array_sum($a) / $n;
        $carry = 0.0;
        foreach ($a as $val) {
            $d = ((double) $val) - $mean;
            $carry += $d * $d;
        }
        if ($sample && $n > 1) --$n;
        return sqrt($carry / $n);
    }
}

// --- PROCESSAMENTO DO FORMULÁRIO ---
$analyzer = new FinancialAnalyzer();
$results = [];
$defaultTickers = "PETR4, VALE3, ITUB4";
$tickersInput = $_POST['tickers'] ?? $defaultTickers;

if ($_SERVER["REQUEST_METHOD"] == "POST" || !empty($tickersInput)) {
    $tickerList = array_map('trim', explode(',', $tickersInput));
    foreach ($tickerList as $t) {
        if (empty($t)) continue;
        
        // Agora retorna um array com ['history'] e ['info']
        $apiResult = $analyzer->getHistoricalData($t);
        
        if ($apiResult && !empty($apiResult['history'])) {
            $metrics = $analyzer->calculateMetrics($apiResult['history']);
            if ($metrics) {
                // Adiciona as informações da empresa às métricas para usar no HTML
                $metrics['company_info'] = $apiResult['info'];
                $results[strtoupper($t)] = $metrics;
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Análise B3 - Risco e Retorno</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Ícones via Phosphor Icons -->
    <script src="https://unpkg.com/@phosphor-icons/web"></script>
    <style>
        .metric-card { transition: all 0.2s; }
        .metric-card:hover { transform: translateY(-5px); }
        .btn-research { transition: background-color 0.2s; }
    </style>
</head>
<body class="bg-slate-900 text-slate-100 font-sans min-h-screen">

    <div class="container mx-auto px-4 py-8">
        <!-- Header -->
        <header class="mb-8 text-center">
            <h1 class="text-4xl font-bold text-emerald-400 mb-2 flex justify-center items-center gap-2">
                <i class="ph ph-trend-up"></i> B3 Analytics by HM Guerra
            </h1>
            <p class="text-slate-400">Calculadora de Riscos, Vantagens e Pesquisa de Mercado</p>
        </header>

        <!-- Search Form -->
        <div class="max-w-xl mx-auto mb-10 bg-slate-800 p-6 rounded-lg shadow-lg border border-slate-700">
            <form method="POST" action="">
                <label class="block text-sm font-medium text-slate-300 mb-2">Códigos das Ações (separados por vírgula)</label>
                <div class="flex gap-2">
                    <input type="text" name="tickers" value="<?php echo htmlspecialchars($tickersInput); ?>" 
                           class="flex-1 bg-slate-900 border border-slate-600 rounded px-4 py-2 text-white focus:outline-none focus:border-emerald-500 placeholder-slate-500" 
                           placeholder="Ex: PETR4, VALE3, WEGE3">
                    <button type="submit" class="bg-emerald-600 hover:bg-emerald-700 text-white font-bold py-2 px-6 rounded transition flex items-center gap-2">
                        <i class="ph ph-magnifying-glass"></i> Analisar
                    </button>
                </div>
                <p class="text-xs text-slate-500 mt-2">* Tenta usar API Brapi.dev. Se falhar, usa simulação.</p>
            </form>
        </div>

        <?php if (!empty($results)): ?>
        
        <!-- Summary Cards -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-10">
            <?php foreach ($results as $ticker => $data): ?>
                <?php 
                    $isPositive = $data['total_return'] >= 0;
                    $riskColor = $data['volatility'] > 30 ? 'text-red-400' : ($data['volatility'] > 20 ? 'text-yellow-400' : 'text-green-400');
                    $info = $data['company_info'];
                ?>
                <div class="metric-card bg-slate-800 rounded-lg border border-slate-700 shadow-xl overflow-hidden flex flex-col">
                    <!-- Card Header -->
                    <div class="p-6 border-b border-slate-700 bg-slate-800/50">
                        <div class="flex justify-between items-start mb-1">
                            <div class="flex items-center gap-3">
                                <?php if($info['logourl']): ?>
                                    <img src="<?php echo $info['logourl']; ?>" alt="Logo" class="w-10 h-10 rounded-full bg-white object-contain p-1">
                                <?php endif; ?>
                                <div>
                                    <h2 class="text-2xl font-bold leading-none"><?php echo $ticker; ?></h2>
                                    <p class="text-xs text-slate-400 mt-1 line-clamp-1" title="<?php echo $info['longName']; ?>">
                                        <?php echo $info['longName']; ?>
                                    </p>
                                </div>
                            </div>
                            <span class="text-xl font-mono font-semibold bg-slate-900 px-2 py-1 rounded border border-slate-700">
                                R$ <?php echo number_format($data['current_price'], 2, ',', '.'); ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Metrics Body -->
                    <div class="p-6 space-y-3 flex-grow">
                        <div class="flex justify-between border-b border-slate-700/50 pb-2">
                            <span class="text-slate-400 text-sm">Vantagem (Retorno)</span>
                            <span class="<?php echo $isPositive ? 'text-emerald-400' : 'text-red-400'; ?> font-bold">
                                <?php echo number_format($data['total_return'], 2, ',', '.'); ?>%
                            </span>
                        </div>
                        <div class="flex justify-between border-b border-slate-700/50 pb-2">
                            <span class="text-slate-400 text-sm">Risco (Volatilidade)</span>
                            <span class="<?php echo $riskColor; ?> font-bold">
                                <?php echo number_format($data['volatility'], 2, ',', '.'); ?>%
                            </span>
                        </div>
                        <div class="flex justify-between border-b border-slate-700/50 pb-2">
                            <span class="text-slate-400 text-sm">Sharpe Ratio</span>
                            <span class="text-blue-400 font-bold">
                                <?php echo number_format($data['sharpe'], 2, ',', '.'); ?>
                            </span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-slate-400 text-sm">Max Drawdown</span>
                            <span class="text-red-500 font-bold">
                                -<?php echo number_format($data['max_drawdown'], 2, ',', '.'); ?>%
                            </span>
                        </div>
                    </div>

                    <!-- Market Research Section -->
                    <div class="bg-slate-900/50 p-4 border-t border-slate-700">
                        <h4 class="text-xs font-bold text-slate-500 uppercase tracking-wide mb-3 flex items-center gap-1">
                            <i class="ph ph-newspaper"></i> Pesquisa de Mercado
                        </h4>
                        
                        <!-- Insight Dinâmico -->
                        <div class="mb-3 text-xs text-slate-300 italic">
                            <?php if ($data['volatility'] > 30): ?>
                                <span class="text-yellow-500"><i class="ph ph-warning"></i> Alta volatilidade.</span> Procure por fatos relevantes recentes ou balanços divulgados.
                            <?php elseif ($data['total_return'] < -10): ?>
                                <span class="text-red-400"><i class="ph ph-trend-down"></i> Tendência de baixa.</span> Verifique notícias sobre o setor da empresa.
                            <?php else: ?>
                                <span class="text-emerald-400"><i class="ph ph-check-circle"></i> Estável.</span> Bom momento para analisar fundamentos de longo prazo.
                            <?php endif; ?>
                        </div>

                        <!-- External Links -->
                        <div class="grid grid-cols-2 gap-2">
                            <a href="https://www.google.com/search?q=<?php echo $ticker; ?>+ações+notícias" target="_blank" 
                               class="btn-research flex items-center justify-center gap-1 bg-slate-700 hover:bg-blue-600 text-xs text-white py-2 rounded">
                                <i class="ph ph-google-logo"></i> Google News
                            </a>
                            <a href="https://statusinvest.com.br/acoes/<?php echo strtolower($ticker); ?>" target="_blank" 
                               class="btn-research flex items-center justify-center gap-1 bg-slate-700 hover:bg-yellow-600 text-xs text-white py-2 rounded">
                                <i class="ph ph-chart-line-up"></i> StatusInvest
                            </a>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Charts Area -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <div class="bg-slate-800 p-4 rounded-lg shadow-lg border border-slate-700">
                <h3 class="text-lg font-bold mb-4 text-center text-slate-200">Evolução do Preço (1 Ano)</h3>
                <canvas id="priceChart"></canvas>
            </div>
            <div class="bg-slate-800 p-4 rounded-lg shadow-lg border border-slate-700">
                <h3 class="text-lg font-bold mb-4 text-center text-slate-200">Comparativo Risco x Retorno</h3>
                <canvas id="riskReturnChart"></canvas>
            </div>
        </div>

        <script>
            // Data Preparation for JS
            const results = <?php echo json_encode($results); ?>;
            const tickers = Object.keys(results);
            
            // --- Price Chart Config ---
            const priceCtx = document.getElementById('priceChart').getContext('2d');
            
            const datasets = tickers.map((ticker, index) => {
                const color = `hsl(${index * 360 / tickers.length}, 70%, 60%)`;
                return {
                    label: ticker,
                    data: results[ticker].data_points.map(d => d.price),
                    borderColor: color,
                    backgroundColor: 'transparent',
                    borderWidth: 2,
                    pointRadius: 0,
                    tension: 0.1
                };
            });

            // Need to unify labels if dates differ, but for simplicity we take the first ticker's dates
            const labels = results[tickers[0]].data_points.map(d => {
                const parts = d.date.split('-'); 
                const date = new Date(parts[0], parts[1] - 1, parts[2]); 
                return date.toLocaleDateString('pt-BR');
            });

            new Chart(priceCtx, {
                type: 'line',
                data: { labels: labels, datasets: datasets },
                options: {
                    responsive: true,
                    interaction: { mode: 'index', intersect: false },
                    scales: {
                        x: { grid: { color: '#334155' }, ticks: { color: '#94a3b8' } },
                        y: { grid: { color: '#334155' }, ticks: { color: '#94a3b8' } }
                    },
                    plugins: { legend: { labels: { color: '#e2e8f0' } } }
                }
            });

            // --- Risk vs Return Scatter Chart ---
            const scatterCtx = document.getElementById('riskReturnChart').getContext('2d');
            
            const scatterData = tickers.map((ticker, index) => {
                const color = `hsl(${index * 360 / tickers.length}, 70%, 60%)`;
                return {
                    label: ticker,
                    data: [{
                        x: results[ticker].volatility,
                        y: results[ticker].total_return
                    }],
                    backgroundColor: color,
                    borderColor: '#fff',
                    pointRadius: 8
                };
            });

            new Chart(scatterCtx, {
                type: 'scatter',
                data: { datasets: scatterData },
                options: {
                    responsive: true,
                    scales: {
                        x: { 
                            title: { display: true, text: 'Risco (Volatilidade %)', color: '#cbd5e1' },
                            grid: { color: '#334155' }, ticks: { color: '#94a3b8' }
                        },
                        y: { 
                            title: { display: true, text: 'Vantagem (Retorno %)', color: '#cbd5e1' },
                            grid: { color: '#334155' }, ticks: { color: '#94a3b8' }
                        }
                    },
                    plugins: { 
                        legend: { labels: { color: '#e2e8f0' } },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': Risco ' + context.parsed.x.toFixed(2) + '%, Retorno ' + context.parsed.y.toFixed(2) + '%';
                                }
                            }
                        }
                    }
                }
            });
        </script>

        <?php endif; ?>
    </div>
</body>
</html>